import { DEFAULT_WHITELIST } from "../config.js";

const $ = (id) => document.getElementById(id);

function normalizeOne(s) {
  return String(s ?? "").trim().toLowerCase();
}

function isValidRule(rule) {
  // 很宽松：只要不是空/空格
  // 你想更严格（域名校验）我也可以给你加
  return !!rule && rule.length <= 255;
}

function setStatus(msg) {
  $("status").textContent = msg || "";
  if (msg) setTimeout(() => ($("status").textContent = ""), 2000);
}

/** ---------- Tabs ---------- */
function initTabs() {
  const tabs = document.querySelectorAll(".tab");
  const panels = {
    custom: $("panel-custom"),
    default: $("panel-default"),
    
  };

  tabs.forEach((t) => {
    t.addEventListener("click", () => {
      tabs.forEach((x) => x.classList.remove("active"));
      t.classList.add("active");
      Object.values(panels).forEach((p) => p.classList.remove("active"));
      panels[t.dataset.tab].classList.add("active");
    });
  });
}

/** ---------- Render default list (read-only) ---------- */
function renderDefault() {
  const box = $("defaultList");
  box.innerHTML = "";

  for (const rule of DEFAULT_WHITELIST || []) {
    const row = document.createElement("div");
    row.className = "row";

    const input = document.createElement("input");
    input.value = rule;
    input.disabled = true;

    const btn = document.createElement("button");
    btn.textContent = "默认";
    btn.disabled = true;

    row.appendChild(input);
    row.appendChild(btn);
    box.appendChild(row);
  }
}

/** ---------- Custom list (editable) ---------- */
function createCustomRow(value = "") {
  const row = document.createElement("div");
  row.className = "row";

  const input = document.createElement("input");
  input.placeholder = "例如：*.google.com 或 api.example.com";
  input.value = value;

  const del = document.createElement("button");
  del.className = "danger";
  del.textContent = "删除";
  del.addEventListener("click", () => row.remove());

  row.appendChild(input);
  row.appendChild(del);
  return row;
}

async function loadCustom() {
  const { userWhitelist } = await chrome.storage.local.get({ userWhitelist: [] });

  const box = $("customList");
  box.innerHTML = "";

  const list = Array.isArray(userWhitelist) ? userWhitelist : [];
  if (list.length === 0) {
    box.appendChild(createCustomRow(""));
  } else {
    for (const rule of list) box.appendChild(createCustomRow(rule));
  }
}

function collectCustomList() {
  const rows = $("customList").querySelectorAll(".row input");
  const raw = Array.from(rows).map((i) => normalizeOne(i.value)).filter(Boolean);

  // 去重
  const seen = new Set();
  const out = [];
  for (const r of raw) {
    if (!isValidRule(r)) continue;
    if (!seen.has(r)) {
      seen.add(r);
      out.push(r);
    }
  }
  return out;
}

async function saveCustomOnly() {
  const list = collectCustomList();
  await chrome.storage.local.set({ userWhitelist: list });
  setStatus("已保存");
}

async function saveAndApply() {
  await saveCustomOnly();
  await chrome.runtime.sendMessage({ type: "REFRESH_PAC" });
  setStatus("已生效");
}

/** ---------- Events ---------- */
$("btnAdd").addEventListener("click", () => {
  $("customList").appendChild(createCustomRow(""));
});

$("btnSave").addEventListener("click", () => {
  saveCustomOnly().catch(console.error);
});

$("btnSaveApply").addEventListener("click", () => {
  saveAndApply().catch(console.error);
});

initTabs();
renderDefault();
loadCustom().catch(console.error);
(async () => {
  const r = await chrome.runtime.sendMessage({ type: "GET_PROXY_STATUS" });
  console.log("BG proxy status:", r);
})();
